/*
 ************************************************************************
 * Copyright (C) 2017, Cisco Systems
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 ************************************************************************
 *
 *  File:    delayed_work.c
 *  Author:  Koushik Chakravarty <kouchakr@cisco.com>
 *
 **********************************************************************
 *
 *  This file contains the apis for scheduling a work to be done by kernel
 *  worker a separate context at a delayed time in the future
 *
 **********************************************************************
 */
#include "delayed_work.h"
#include "dbgout.h"
#ifdef NVM_BPF_USERSPACE
#include <string.h>
#else
#include <linux/slab.h>
#include <linux/version.h>
#include <linux/param.h>
#include "defines.h"

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 7, 0))
#define create_rt_workqueue(name) \
	alloc_workqueue("%s", WQ_MEM_RECLAIM | WQ_UNBOUND | WQ_HIGHPRI, \
			 0, (name))
#endif
#endif


#ifdef NVM_BPF_USERSPACE
/*
* Worker thread function that processes delayed work items
*/
static void *delayed_work_thread(void *arg) {
	struct _delayed_workqueue *workq = (struct _delayed_workqueue *)arg;
	TRACE(ERROR, LOG("Delayed work thread started: name %s", workq->name));
	
	while (true) {
		pthread_mutex_lock(&workq->mutex);
		
		while (!workq->shutdown && workq->work_items.empty()) {
			// Wait for work or shutdown signal
			pthread_cond_wait(&workq->notify, &workq->mutex);
		}
		
		if (workq->shutdown) {
			pthread_mutex_unlock(&workq->mutex);
			break;
		}
		
		// Get current time
		time_t now = time(NULL);
		bool work_available = false;
		
		// Find work item ready to execute
		for (auto it = workq->work_items.begin(); it != workq->work_items.end(); ++it) {
			if (now >= it->execute_at) {
				// Item ready to execute
				struct delayed_work_item item = *it;
				workq->work_items.erase(it);
				pthread_mutex_unlock(&workq->mutex);
				
				// Execute work
				if (item.work && item.work->fn_work) {
					item.work->fn_work(item.work->context);
				}
				
				work_available = true;
				break;
			}
		}
		
		if (!work_available) {
			// Calculate time to next work item
			time_t next_time = now + 3600; // Default: 1 hour
			for (auto& item : workq->work_items) {
				if (item.execute_at < next_time) {
					next_time = item.execute_at;
				}
			}
			
			if (next_time > now) {
				// Wait until next work item or notification
				struct timespec ts;
				ts.tv_sec = next_time;
				ts.tv_nsec = 0;
				
				pthread_cond_timedwait(&workq->notify, &workq->mutex, &ts);
			}
			
			pthread_mutex_unlock(&workq->mutex);
		}
	}
	
	return NULL;
}
#else

/*
*   \brief callback to the worker
*/
static void work_handler(struct work_struct *work_struct)
{
	struct _delayed_work *work = NULL;

	if (NULL == work_struct)
		return;
	work =
	    container_of(to_delayed_work(work_struct), struct _delayed_work,
			 d_work);

	if (NULL != work && NULL != work->fn_work)
		work->fn_work(work->context);
}
#endif

/*
*   \brief function to create a delayed work queue with a provided name
*   \param[in] name name of the queue
*   \return true/false
*/
static bool
create_delayed_workqueue(struct _delayed_workqueue *work_q, const char *name)
{
	if (!name || !work_q)
		return false;

#ifdef NVM_BPF_USERSPACE
	strcpy(work_q->name, name);
	work_q->shutdown = false;
	work_q->work_items.clear();
	
	if (pthread_mutex_init(&work_q->mutex, NULL) != 0)
		return false;
		
	if (pthread_cond_init(&work_q->notify, NULL) != 0) {
		pthread_mutex_destroy(&work_q->mutex);
		return false;
	}
	
	// Create single worker thread
	if (pthread_create(&work_q->worker_thread, NULL, delayed_work_thread, work_q) != 0) {
		pthread_mutex_destroy(&work_q->mutex);
		pthread_cond_destroy(&work_q->notify);
		return false;
	}
#else
	work_q->wq = create_rt_workqueue(name);
	if (!work_q->wq)
		return false;
#endif
	return true;
}

/*
*   \brief function to create a delayed work
*   \param[in] fpWorkFn pointer to the worker function
*   \param[in] context user context that needs to be passed to the callback
*   \return true/false
*/
static bool
create_delayed_work(struct _delayed_work *work, fWorkFn_t *fpWorkFn,
		    void *context)
{
	if (NULL == fpWorkFn || NULL == work) {
		return false;
	}

#ifndef NVM_BPF_USERSPACE
	INIT_DELAYED_WORK(&work->d_work, work_handler);
#endif
	work->fn_work = fpWorkFn;
	work->context = context;

	return true;
}

/*
*   \brief function to schedule work on given queue
*   \param[in] pWq workqueue
*   \param[in] work worker struct
*   \param[in] delay_in_secs delay in seconds
*   \return true or false
*/
static bool
schedule_delayed_workon(struct _delayed_workqueue *pWq,
			struct _delayed_work *work,
			unsigned long delay_in_secs)
{
#ifdef NVM_BPF_USERSPACE
	if (NULL == pWq || NULL == work)
		return false;
		
	pthread_mutex_lock(&pWq->mutex);
	
	if (pWq->shutdown) {
		pthread_mutex_unlock(&pWq->mutex);
		return false;
	}
	
	// Create work item
	struct delayed_work_item item;
	item.work = work;
	item.delay_secs = delay_in_secs;
	item.execute_at = time(NULL) + delay_in_secs;
	
	// Add to queue
	pWq->work_items.push_back(item);
	
	// Signal worker thread
	pthread_cond_signal(&pWq->notify);
	pthread_mutex_unlock(&pWq->mutex);
	
	return true;
#else
	if (NULL != work && (NULL != pWq && NULL != pWq->wq)) {
		/*
		 * This api takes the delay in jiffies which is a counter
		 * incremented every clock tick.
		 * In a second there are HZ number of clock ticks.
		 * HZ is a symbol defined in <linux/param.h>
		 */
		return queue_delayed_work(pWq->wq, &work->d_work,
					  (delay_in_secs * HZ));
	}
#endif
	return false;
}

/*
*   \brief function to Destroy worker
*   \param[in] work worker struct
*   \return
*/
static void destroy_delayed_work(struct _delayed_work *work)
{
#ifndef NVM_BPF_USERSPACE
	if (NULL != work)
		cancel_delayed_work(&work->d_work);
#endif
}

/*
*   \brief function to Destroy work queue
*   \param[in] pWq workqueue
*   \return
*/
static void destroy_delayed_workqueue(struct _delayed_workqueue *pWq)
{
#ifdef NVM_BPF_USERSPACE
	if (NULL != pWq) {
		// Signal shutdown
		pthread_mutex_lock(&pWq->mutex);
		pWq->shutdown = true;
		pthread_cond_signal(&pWq->notify);
		pthread_mutex_unlock(&pWq->mutex);
		
		// Wait for thread to terminate
		pthread_join(pWq->worker_thread, NULL);
		
		// Clean up resources
		pthread_mutex_destroy(&pWq->mutex);
		pthread_cond_destroy(&pWq->notify);
		pWq->work_items.clear();
	}
#else
	if (NULL != pWq && NULL != pWq->wq)
		destroy_workqueue(pWq->wq);
#endif
}

/*
 * Create a work and associated queue
 */
bool
create_delayed_work_on_queue(struct delayed_work_on_q *work,
			     const char *name, fWorkFn_t *fpWorkFn,
			     void *context)
{
	if (NULL == work)
		return false;

	if (true != create_delayed_work(&work->work, fpWorkFn, context))
		return false;

	if (true != create_delayed_workqueue(&work->work_queue, name)) {
		destroy_delayed_work(&work->work);
		return false;
	}
	return true;
}

/*
 * Schedule work
 */
bool
schedule_delayed_work_on_queue(struct delayed_work_on_q *workqueue,
			       unsigned long delayInSecs)
{
	if (NULL == workqueue)
		return false;

	return schedule_delayed_workon(&workqueue->work_queue, &workqueue->work,
				       delayInSecs);
}

/*
 * Destroy queue
 */
void destroy_delayed_work_on_queue(struct delayed_work_on_q *workqueue)
{
	if (NULL != workqueue) {
		destroy_delayed_work(&workqueue->work);
		destroy_delayed_workqueue(&workqueue->work_queue);
	}
}